/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Implemnation of the "Packet send/receive" dialog

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2008-11-26  Wrong data size will be now adjusted

    2006-06-28  initial version

**************************************************************************************/

#include "stdafx.h"
#include "cifXTest.h"
#include "PacketDlg.h"
#include "CifXDeviceBase.h"
#include "Cifxtestdlg.h"
#include ".\packetdlg.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// \file PacketDlg.cpp
///  Implemnation of the "Packet send/receive" dialog
///////////////////////////////////////////////////////////////////////////////////////////

#define CIFX_RECV_TIMEOUT 10
#define CIFX_SEND_TIMEOUT 10

#define PACKET_POLL_TIMEOUT 10

IMPLEMENT_DYNAMIC(CPacketDlg, CBaseDialog)


/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent window
/////////////////////////////////////////////////////////////////////////////
CPacketDlg::CPacketDlg(CWnd* pParent /*=NULL*/)
  : CBaseDialog(CPacketDlg::IDD, pParent)
  , m_fAutoIncrementID(false)
  , m_fSendCyclic(false)
  , m_fRecvCyclic(false)
  , m_ulSendCounter(0)
  , m_ulRecvCounter(0)
  , m_fDeviceAccess(false)
{
  ZeroMemory(&m_tSendPacket, sizeof(m_tSendPacket));
  ZeroMemory(&m_tRecvPacket, sizeof(m_tRecvPacket));
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CPacketDlg::~CPacketDlg()
{
  m_fRecvCyclic = false;
  m_fSendCyclic = false;
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV support
///   \param pDX
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::DoDataExchange(CDataExchange* pDX)
{
  CBaseDialog::DoDataExchange(pDX);

  DDX_Control(pDX, IDC_SEND_DEST, m_cSendDestCtrl);
  DDX_Control(pDX, IDC_SEND_SRC,  m_cSendSrcCtrl);
  DDX_Control(pDX, IDC_SEND_DESTID, m_cSendDestIDCtrl);
  DDX_Control(pDX, IDC_SEND_SRCID,  m_cSendSrcIDCtrl);
  DDX_Control(pDX, IDC_SEND_ID,    m_cSendIDCtrl);
  DDX_Control(pDX, IDC_SEND_STATE, m_cSendStateCtrl);
  DDX_Control(pDX, IDC_SEND_CMD,   m_cSendCmdCtrl);
  DDX_Control(pDX, IDC_SEND_EXT,   m_cSendExtCtrl);
  DDX_Control(pDX, IDC_SEND_ROUTE, m_cSendRoutCtrl);

  DDX_Control(pDX, IDC_RECV_DEST,  m_cRecvDestCtrl);
  DDX_Control(pDX, IDC_RECV_SRC,   m_cRecvSrcCtrl);
  DDX_Control(pDX, IDC_RECV_DESTID,m_cRecvDestIDCtrl);
  DDX_Control(pDX, IDC_RECV_SRCID, m_cRecvSrcIDCtrl);
  DDX_Control(pDX, IDC_RECV_LEN,   m_cRecvLenCtrl);
  DDX_Control(pDX, IDC_RECV_ID,    m_cRecvIDCtrl);
  DDX_Control(pDX, IDC_RECV_STATE, m_cRecvStateCtrl);
  DDX_Control(pDX, IDC_RECV_CMD,   m_cRecvCmdCtrl);
  DDX_Control(pDX, IDC_RECV_EXT,   m_cRecvExtCtrl);
  DDX_Control(pDX, IDC_RECV_ROUTE, m_cRecvRoutCtrl);

  DDX_Control(pDX, IDC_CHK_SEND_AUTOID, m_cSendAutoIdCtrl);
  DDX_Control(pDX, IDC_CHK_SENDCYCLIC,  m_cSendCyclicCtrl);
  DDX_Control(pDX, IDC_CHK_RECVCYCLIC,  m_cRecvCyclicCtrl);
  DDX_Control(pDX, IDC_SEND_DATA,       m_cSendDataCtrl);
}


BEGIN_MESSAGE_MAP(CPacketDlg, CBaseDialog)
  ON_BN_CLICKED(IDC_BTN_GETPACKET, OnBnClickedBtnGetpacket)
  ON_BN_CLICKED(IDC_BTN_PUTPACKET, OnBnClickedBtnPutpacket)
  ON_BN_CLICKED(IDC_BTN_RESETRECV, OnBnClickedBtnResetrecv)
  ON_BN_CLICKED(IDC_BTN_RESETSEND, OnBnClickedBtnResetsend)
  ON_BN_CLICKED(IDC_CHK_SEND_AUTOID, OnBnClickedChkSendAutoid)
  ON_BN_CLICKED(IDC_CHK_SENDCYCLIC, OnBnClickedChkSendcyclic)
  ON_BN_CLICKED(IDC_CHK_RECVCYCLIC, OnBnClickedChkRecvcyclic)
  ON_EN_KILLFOCUS(IDC_SEND_DATA, OnEnKillfocusSendData)
  ON_EN_UPDATE(IDC_SEND_DATA, OnEnUpdateSendData)
  ON_MESSAGE(WM_UPDATE_COUNTERS, UpdateCounters)
  ON_MESSAGE(WM_UPDATE_SEND,     UpdateSendPacket)
  ON_MESSAGE(WM_UPDATE_RECV,     UpdateReceivePacket)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
/// First time dialog initialization. Links the edit controls with the header structures
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CPacketDlg::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  m_cSendDestCtrl.SetUpdateVariable(  &m_tSendPacket.tHeader.ulDest);
  m_cSendSrcCtrl.SetUpdateVariable(   &m_tSendPacket.tHeader.ulSrc);
  m_cSendDestIDCtrl.SetUpdateVariable(&m_tSendPacket.tHeader.ulDestId);
  m_cSendSrcIDCtrl.SetUpdateVariable( &m_tSendPacket.tHeader.ulSrcId);
  m_cSendIDCtrl.SetUpdateVariable(    &m_tSendPacket.tHeader.ulId);
  m_cSendStateCtrl.SetUpdateVariable( &m_tSendPacket.tHeader.ulState);
  m_cSendCmdCtrl.SetUpdateVariable(   &m_tSendPacket.tHeader.ulCmd);
  m_cSendExtCtrl.SetUpdateVariable(   &m_tSendPacket.tHeader.ulExt);
  m_cSendRoutCtrl.SetUpdateVariable(  &m_tSendPacket.tHeader.ulRout);

/*
  m_cFixedFont.CreatePointFont(80, _T("Courier new"));

  GetDlgItem(IDC_SEND_DATA)->SetFont(&m_cFixedFont);
  GetDlgItem(IDC_RECV_DATA)->SetFont(&m_cFixedFont);
*/
  UpdateCounters();

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Get Packet button click event (fetches a single packet)
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedBtnGetpacket()
{
  GetPacket(1);
}

/////////////////////////////////////////////////////////////////////////////
/// Put Packet button click event (sends a single packet)
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedBtnPutpacket()
{
  if (ValidateSendData())
  {
    long lSendError = PutPacket();

    if(CIFX_NO_ERROR != lSendError)
    {
      CString csError;
      csError.Format(_T("Error sending packet to device!\n") \
                      _T("Error \t\t: 0x%08X\n") \
                      _T("Description \t: %s\n"),
                      lSendError,
                      (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lSendError));

      AfxMessageBox(csError, MB_OK | MB_ICONERROR);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Reset Receive button click event (resets receive packet counter)
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedBtnResetrecv()
{
  m_ulRecvCounter = 0;
  UpdateCounters();
}

/////////////////////////////////////////////////////////////////////////////
/// Reset Send button click event (resets send packet counter)
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedBtnResetsend()
{
  m_ulSendCounter = 0;
  UpdateCounters();
}

/////////////////////////////////////////////////////////////////////////////
/// Updates the receive packet view, including the data portion
/////////////////////////////////////////////////////////////////////////////
LRESULT CPacketDlg::UpdateReceivePacket(WPARAM /*wParam = 0*/, LPARAM /*lParam = 0*/)
{
  m_cRecvDestCtrl.SetValue(   m_tRecvPacket.tHeader.ulDest);
  m_cRecvSrcCtrl.SetValue(    m_tRecvPacket.tHeader.ulSrc);
  m_cRecvDestIDCtrl.SetValue( m_tRecvPacket.tHeader.ulDestId);
  m_cRecvSrcIDCtrl.SetValue(  m_tRecvPacket.tHeader.ulSrcId);
  m_cRecvIDCtrl.SetValue(     m_tRecvPacket.tHeader.ulId);
  m_cRecvStateCtrl.SetValue(  m_tRecvPacket.tHeader.ulState);
  m_cRecvCmdCtrl.SetValue(    m_tRecvPacket.tHeader.ulCmd);
  m_cRecvExtCtrl.SetValue(    m_tRecvPacket.tHeader.ulExt);
  m_cRecvRoutCtrl.SetValue(   m_tRecvPacket.tHeader.ulRout);

  TCHAR szBuffer[12] = {0};
  m_cRecvLenCtrl.SetWindowText( _ltot(m_tRecvPacket.tHeader.ulLen, szBuffer, 10));

  CString csRecvData;
  for(unsigned long ulIdx = 0; ulIdx < m_tRecvPacket.tHeader.ulLen; ++ulIdx)
  {
    // The returned message is bigger than out buffer, break out of loop to prevent access violation
    if(ulIdx >= sizeof(m_tRecvPacket.abData))
      break;

    if( (ulIdx > 0) &&
        ((ulIdx % 16) == 0) )
      csRecvData.Append(_T("\r\n"));

    csRecvData.AppendFormat(_T("%02X "), m_tRecvPacket.abData[ulIdx]);
  }

  GetDlgItem(IDC_RECV_DATA)->SetWindowText(csRecvData);

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Updates the send packet view (ID currently only, as all other are only updated by user)
/////////////////////////////////////////////////////////////////////////////
LRESULT CPacketDlg::UpdateSendPacket(WPARAM /*wParam = 0*/, LPARAM /*lParam = 0*/)
{
  m_cSendIDCtrl.SetValue( m_tSendPacket.tHeader.ulId);

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Updates the send/receive counter controls
/////////////////////////////////////////////////////////////////////////////
LRESULT CPacketDlg::UpdateCounters(WPARAM /*wParam = 0*/, LPARAM /*lParam = 0*/)
{
  TCHAR szBuffer[12] = {0};

  GetDlgItem(IDC_RECV_COUNTER)->SetWindowText( _ltot(m_ulRecvCounter, szBuffer, 10));
  GetDlgItem(IDC_SEND_COUNTER)->SetWindowText( _ltot(m_ulSendCounter, szBuffer, 10));

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Auto ID incrementation checkbox click event
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedChkSendAutoid()
{
  m_fAutoIncrementID = m_cSendAutoIdCtrl.GetCheck() == BST_CHECKED;
}

/////////////////////////////////////////////////////////////////////////////
/// Cyclic send checkbox click event
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedChkSendcyclic()
{
  m_fSendCyclic = m_cSendCyclicCtrl.GetCheck() == BST_CHECKED;
  GetDlgItem(IDC_BTN_PUTPACKET)->EnableWindow(!m_fSendCyclic);
}

/////////////////////////////////////////////////////////////////////////////
/// Cyclic receive checkbox click event
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnBnClickedChkRecvcyclic()
{
  m_fRecvCyclic = m_cRecvCyclicCtrl.GetCheck() == BST_CHECKED;
  GetDlgItem(IDC_BTN_GETPACKET)->EnableWindow(!m_fRecvCyclic);
}

/////////////////////////////////////////////////////////////////////////////
/// Class function handling all cyclic events
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::ThreadFunction(void)
{
  m_fDeviceAccess = true;

  if(m_fRecvCyclic)
  {
    GetPacket();
  }

  if(m_fSendCyclic)
  {
    long lSendError = PutPacket();

    if(CIFX_NO_ERROR != lSendError)
    {
      CString csError;
      csError.Format(_T("Error sending packet to device!\n") \
                     _T("Error \t\t: 0x%08X\n") \
                     _T("Description \t: %s\n")\
                     _T("\n") \
                     _T("Cyclic send will be aborted!"),
                     lSendError,
                     (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lSendError));

      AfxMessageBox(csError, MB_OK | MB_ICONERROR);

      //Asynchronous signalling, to prevent message loop blocking
      m_cSendCyclicCtrl.PostMessage(BM_SETCHECK, BST_UNCHECKED, 0);
      m_fSendCyclic = false;
      GetDlgItem(IDC_BTN_PUTPACKET)->EnableWindow(TRUE);
    }
  }

  PostMessage(WM_UPDATE_COUNTERS); //replaces UpdateCounters();

  m_fDeviceAccess = false;
}

/////////////////////////////////////////////////////////////////////////////
/// Sends a single packet to the device and updates counters/packet header
/// according to the current setup
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
long CPacketDlg::PutPacket(void)
{
  if(NULL == CcifXTestDlg::s_pcDevice)
    return CIFX_DRV_CHANNEL_NOT_INITIALIZED;

  long lRet = CcifXTestDlg::s_pcDevice->PutPacket(&m_tSendPacket, CIFX_SEND_TIMEOUT);

  if(m_fAutoIncrementID)
    ++m_tSendPacket.tHeader.ulId;

  ASSERT(CIFX_NO_ERROR == lRet);
  if(lRet == CIFX_NO_ERROR)
  {
    ++m_ulSendCounter;
  } else
  {
    if(m_fAutoIncrementID)
      --m_tSendPacket.tHeader.ulId;
  }

  PostMessage(WM_UPDATE_SEND); //replaces UpdateSendPacket();

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Retrieves a single packet from the device and updates counters/packet header
/// according to the current setup
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
long CPacketDlg::GetPacket(ULONG ulMaxRecv)
{
  if(NULL == CcifXTestDlg::s_pcDevice)
    return CIFX_DRV_CHANNEL_NOT_INITIALIZED;

  uint32_t ulRecvCount = 0;
  uint32_t ulSendCount = 0;

  long lRet = CcifXTestDlg::s_pcDevice->GetMBXState(&ulRecvCount, &ulSendCount);

  if(ulMaxRecv != 0)
    ulRecvCount = min(ulMaxRecv, ulRecvCount);

  if(CIFX_NO_ERROR == lRet)
  {
    while(ulRecvCount-- > 0)
    {
      lRet = CcifXTestDlg::s_pcDevice->GetPacket(&m_tRecvPacket, sizeof(m_tRecvPacket), 0);

      if(lRet == CIFX_NO_ERROR)
      {
        if(m_ulRecvCounter != 0)
        {
          if( m_fAutoIncrementID &&
              (m_tTempRecvPacket.tHeader.ulId == m_tRecvPacket.tHeader.ulId) )
          {
            CString csError;

            csError.Format(_T("Duplicate Message ID found (0x%08X)"), m_tTempRecvPacket.tHeader.ulId);
            AfxMessageBox(csError);
          }
        }

        m_tTempRecvPacket = m_tRecvPacket;

        ++m_ulRecvCounter;
      }
    }
    PostMessage(WM_UPDATE_RECV); //replaces UpdateReceivePacket();
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Validate send data
/////////////////////////////////////////////////////////////////////////////
bool CPacketDlg::ValidateSendData()
{
  bool fRet = false;

  // Check if device still exists
  if( NULL == CcifXTestDlg::s_pcDevice) return fRet;

  CString csTemp;
  m_cSendDataCtrl.GetWindowText(csTemp);

  int iDataLen = 0;

  // remove all whitespace characters
  csTemp.Remove(_T(' '));
  csTemp.Remove(_T('\r'));
  csTemp.Remove(_T('\n'));


  int iStrLen = csTemp.GetLength();

  if(iStrLen % 2)
  {
    //invalid string length, one nibble must be missing
    AfxMessageBox(  _T("The data you entered is missing one nibble!\r\n") \
                    _T("Length will be adjusted!"));

    // Cut of data to valid length
    csTemp = csTemp.Left( iStrLen - 1);
    m_cSendDataCtrl.SetWindowText(csTemp);

    m_cSendDataCtrl.SetSel(-1, -1);
    m_cSendDataCtrl.SetFocus();

  } else
  {
    int iByteCount = iStrLen / 2;

    if( (unsigned long)(iByteCount + CIFX_PACKET_HEADER_SIZE) > CcifXTestDlg::s_pcDevice->GetMailboxSize())
    {
      //packet does not fit into device mailbox
      CString csError;
      csError.Format( _T("The data you entered does not fit the device's mailbox!\n\r")\
                      _T("Mailboxsize = %u, Your data size (including packet header) = %u.\r\n") \
                      _T("Length will be adjusted!") ,
                      CcifXTestDlg::s_pcDevice->GetMailboxSize(),
                      iByteCount + CIFX_PACKET_HEADER_SIZE);

      AfxMessageBox(csError);

      // Cut of data to valid length
      csTemp = csTemp.Left( (CcifXTestDlg::s_pcDevice->GetMailboxSize() - CIFX_PACKET_HEADER_SIZE) * 2);
      m_cSendDataCtrl.SetWindowText(csTemp);

      m_cSendDataCtrl.SetSel(-1, -1);
      m_cSendDataCtrl.SetFocus();

    } else
    {
      CString csOutput;

      //parse string
      for(int iByteIdx = 0; iByteIdx < iByteCount; ++iByteIdx)
      {
        TCHAR* pszEnd = NULL;
        m_tSendPacket.abData[iByteIdx] = (unsigned char)_tcstol(csTemp.Mid(iByteIdx * 2, 2), &pszEnd, 16);

        if((iByteIdx > 0) && ((iByteIdx % 16) == 0))
          csOutput.Append(_T("\r\n"));

        csOutput.AppendFormat(_T("%02X "), m_tSendPacket.abData[iByteIdx]);
      }

      m_cSendDataCtrl.SetWindowText(csOutput);
      iDataLen = iByteCount;
      m_tSendPacket.tHeader.ulLen = (unsigned long)iDataLen;

      fRet = true;
    }
  }

  TCHAR szBuffer[12] = {0};
  GetDlgItem(IDC_SEND_LEN)->SetWindowText( _ltot(iDataLen, szBuffer, 10));

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// User finished Send Data editing event
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnEnKillfocusSendData()
{
  ValidateSendData();
}

/////////////////////////////////////////////////////////////////////////////
/// User updated Send Data content event
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnEnUpdateSendData()
{
  //TODO: Filter invalid characters
}

/////////////////////////////////////////////////////////////////////////////
/// Device has changed or Dialog has been activated
/////////////////////////////////////////////////////////////////////////////
void CPacketDlg::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  if(NULL == pcDevice)
    while(m_fDeviceAccess)
      Sleep(0);
}